#!/bin/bash
# Filename   : cve-2021-26084-update.sh
# Description: Temporary workaround for CVE-2021-26084 for Confluence instances running on Linux based Operating Systems
# Reference  : https://confluence.atlassian.com/display/DOC/Confluence+Security+Advisory+-+2021-08-25
# Usage      : sh cve-2021-26084-update.sh
# Version    : 1.4
set -u

# ###########################################
# Update user specific data in this section

# set this to where Confluence is installed
# e.g. INSTALLATION_DIRECTORY=/opt/atlassian/confluence
INSTALLATION_DIRECTORY=


# ########################################
# Do not change anything below this line

if [ -z "$INSTALLATION_DIRECTORY" ]
then
    echo "Please set INSTALLATION_DIRECTORY within this script and try running this script again.";
    exit 1;
fi

# Make sure we are running as the correct Linux user
if [ ! -w "$INSTALLATION_DIRECTORY/confluence" ]
then
    echo "ERROR: Please run this script as the Linux user that owns the $INSTALLATION_DIRECTORY/confluence directory"
    echo " (i.e." `ls -ld "$INSTALLATION_DIRECTORY/confluence" | awk '{ print $3 }'`")";
    exit 1;
fi

# Change SED flags dependent on OS
SEDFLAGS=-ri.bak
if uname -a | grep -qi "Darwin"
then
    SEDFLAGS=-Ei.bak
fi

# Change to Install Directory
echo "chdir '$INSTALLATION_DIRECTORY'"
cd "$INSTALLATION_DIRECTORY";
if [ $? -ne 0 ]; then
    echo "ERROR: Failed to change to the directory $INSTALLATION_DIRECTORY!"
    exit 1;
fi
echo ""

# check zip/unzip dependencies up front
UNZIP=`which unzip`
ZIP=`which zip`
if [ -z "$ZIP" ]
then
    echo "ERROR: 'zip' package is missing. Please install 'zip' and try running this update script again.";
    echo "e.g. RHEL based OS         , try 'sudo yum install zip unzip'"
    echo "e.g. Ubuntu/Docker based OS, try (as root) 'apt update; apt install zip unzip'"
    echo "UPDATE FAILED, EXITING!"
    echo ""
    exit 1;
fi
if [ -z "$UNZIP" ]
then
    echo "ERROR: 'unzip' package is missing. Please install 'unzip' and try running this update script again.";
    echo "e.g. RHEL based OS         , try 'sudo yum install zip unzip'"
    echo "e.g. Ubuntu/Docker based OS, try (as root) 'apt update; apt install zip unzip'"
    echo "UPDATE FAILED, EXITING!"
    echo ""
    exit 1;
fi

# ######################################
# File 1 of 5

echo "File 1: 'confluence/users/user-dark-features.vm':"
echo -n "   a. backing up file.. "
cp -np confluence/users/user-dark-features.vm confluence/users/user-dark-features.vm.original;
echo "done"
echo -n "   b. updating file.. "
sed $SEDFLAGS 's/(Enable dark feature.+value=)[^"]+"/\1featureKey"/' confluence/users/user-dark-features.vm;
echo "done"
echo "   c. showing file changes.."
diff confluence/users/user-dark-features.vm.original confluence/users/user-dark-features.vm;
echo -n "   d. validating file changes.. "
if grep -qi "'\$!action.featureKey'" confluence/users/user-dark-features.vm
then
    echo "ERROR: Failed to update confluence/users/user-dark-features.vm"
    echo ""
    echo "UPDATE FAILED, EXITING!"
    exit 1;
fi
if ! grep -qi "value=featureKey" confluence/users/user-dark-features.vm
then
    echo "ERROR: Failed to update confluence/users/user-dark-features.vm"
    echo ""
    echo "UPDATE FAILED, EXITING!"
    exit 1;
fi
echo "ok"
echo "   e. file updated successfully!"
echo ""

# ######################################
# File 2 of 5

echo "File 2: 'confluence/login.vm':"
echo -n "   a. backing up file.. "
cp -np confluence/login.vm confluence/login.vm.original;
echo "done"
echo -n "   b. updating file.. "
sed $SEDFLAGS 's/("Hidden" "name=.token." "value=)[^"]+"/\1token"/' confluence/login.vm;
echo "done"
echo "   c. showing file changes.."
diff confluence/login.vm.original confluence/login.vm
echo -n "   d. validating file changes.. "
if grep -qi "'\$!action.token'" confluence/login.vm
then
    echo "ERROR: Failed to update confluence/login.vm"
    echo ""
    echo "UPDATE FAILED, EXITING!"
    exit 1;
fi
if ! grep -qi "value=token" confluence/login.vm
then
    echo "ERROR: Failed to update confluence/login.vm"
    echo ""
    echo "UPDATE FAILED, EXITING!"
    exit 1;
fi
echo "ok"
echo "   e. file updated successfully!"
echo ""

# ######################################
# File 3 of 5

echo "File 3: 'confluence/pages/createpage-entervariables.vm':"
echo -n "   a. backing up file.. "
cp -np confluence/pages/createpage-entervariables.vm confluence/pages/createpage-entervariables.vm.original;
echo "done"
echo -n "   b. updating file.. "
sed $SEDFLAGS 's/("Hidden" "name=.([a-zA-Z]+)." "value=).\$[!l][^"]+"/\1\2"/' confluence/pages/createpage-entervariables.vm;
echo "done"
echo "   c. showing file changes.."
diff confluence/pages/createpage-entervariables.vm.original confluence/pages/createpage-entervariables.vm
echo -n "   d. validating file changes.."
if grep -qi "value='\$!querystring" confluence/pages/createpage-entervariables.vm
then
    echo "ERROR: Failed to update confluence/pages/createpage-entervariables.vm"
    echo ""
    echo "UPDATE FAILED, EXITING!"
    exit 1;
fi
if grep -qi "value='\$linkCreation" confluence/pages/createpage-entervariables.vm
then
    echo "ERROR: Failed to update confluence/pages/createpage-entervariables.vm"
    echo ""
    echo "UPDATE FAILED, EXITING!"
    exit 1;
fi
if ! grep -qi "value=linkCreation" confluence/pages/createpage-entervariables.vm
then
    echo "ERROR: Failed to update confluence/pages/createpage-entervariables.vm"
    echo ""
    echo "UPDATE FAILED, EXITING!"
    exit 1;
fi
echo "ok"
echo "   e. file updated successfully!"
echo ""

# ######################################
# File 4 of 5

echo "File 4: 'confluence/template/custom/content-editor.vm':"
echo -n "   a. backing up file.. "
cp -np confluence/template/custom/content-editor.vm confluence/template/custom/content-editor.vm.original;
echo "done"
echo -n "   b. updating file.. "
sed $SEDFLAGS 's/("Hidden" "name=.([a-zA-Z]+)." "value=).\$[!l][^"]+"/\1\2"/' confluence/template/custom/content-editor.vm;
sed $SEDFLAGS 's/("Hidden" "id=sourceTemplateId.*value=)[^"]+"/\1templateId"/' confluence/template/custom/content-editor.vm;
echo "done"
echo "   c. showing file changes.."
diff confluence/template/custom/content-editor.vm.original confluence/template/custom/content-editor.vm
echo "   d. file updated successfully!"
echo ""

# ######################################
# File 5 of 5

CONFLUENCE_EDITOR_JAR=`ls -1 confluence/WEB-INF/atlassian-bundled-plugins/confluence-editor-loader*.jar 2> /dev/null`
echo "File 5: 'confluence/WEB-INF/atlassian-bundled-plugins/confluence-editor-loader*.jar':"
if [ ! -z "$CONFLUENCE_EDITOR_JAR" ]
then
    echo "   a. extracting templates/editor-preload-container.vm from $CONFLUENCE_EDITOR_JAR.. "
    export TMP_EXTRACT_DIR=.
    unzip -o -d $TMP_EXTRACT_DIR $CONFLUENCE_EDITOR_JAR templates/editor-preload-container.vm;
    if [ -f templates/editor-preload-container.vm ]
    then
        cp -np $TMP_EXTRACT_DIR/templates/editor-preload-container.vm $TMP_EXTRACT_DIR/templates/editor-preload-container.vm.original;

        echo -n "   b. updating file.. "
        sed $SEDFLAGS 's/("Hidden" "id=syncRev.*value=)[^"]+"/\1syncRev"/' $TMP_EXTRACT_DIR/templates/editor-preload-container.vm;
        echo "done"
        echo "   c. showing file changes.."
        diff $TMP_EXTRACT_DIR/templates/editor-preload-container.vm.original $TMP_EXTRACT_DIR/templates/editor-preload-container.vm;

        echo -n "   d. validating file changes.. "
        if grep -qi "action.syncRev" $TMP_EXTRACT_DIR/templates/editor-preload-container.vm
        then
            echo "ERROR: Failed to update $TMP_EXTRACT_DIR/templates/editor-preload-container.vm"
            echo ""
            echo "UPDATE FAILED, EXITING!"
            exit 1;
        fi
        if ! grep -qi "value=syncRev" $TMP_EXTRACT_DIR/templates/editor-preload-container.vm
        then
            echo "ERROR: Failed to update $TMP_EXTRACT_DIR/templates/editor-preload-container.vm"
            echo ""
            echo "UPDATE FAILED, EXITING!"
            exit 1;
        fi
        echo "ok"

        echo -n "   e. updating $CONFLUENCE_EDITOR_JAR with $TMP_EXTRACT_DIR/templates/editor-preload-container.vm.."
        zip "$CONFLUENCE_EDITOR_JAR" $TMP_EXTRACT_DIR/templates/editor-preload-container.vm;
        ls -l "$CONFLUENCE_EDITOR_JAR";

        echo -n "   f. cleaning up temp files.."
        rm -f $TMP_EXTRACT_DIR/templates/editor-preload-container.vm $TMP_EXTRACT_DIR/templates/editor-preload-container.vm.bak $TMP_EXTRACT_DIR/templates/editor-preload-container.vm.original;
        echo "ok"

        echo "   g. extracting templates/editor-preload-container.vm from $CONFLUENCE_EDITOR_JAR again to check changes within JAR.. "
        export TMP_EXTRACT_DIR=.
        unzip -o -d $TMP_EXTRACT_DIR $CONFLUENCE_EDITOR_JAR templates/editor-preload-container.vm;
        if [ ! -f templates/editor-preload-container.vm ]
        then
            echo "ERROR: Failed to extract templates/editor-preload-container.vm from $CONFLUENCE_EDITOR_JAR"
            echo ""
            echo "UPDATE FAILED, EXITING!"
            exit 1;
        fi

        echo -n "   h. validating file changes for file within updated JAR.. "
        if grep -qi "action.syncRev" $TMP_EXTRACT_DIR/templates/editor-preload-container.vm
        then
            echo "ERROR: Failed to update $TMP_EXTRACT_DIR/templates/editor-preload-container.vm"
            echo ""
            echo "UPDATE FAILED, EXITING!"
            exit 1;
        fi
        if ! grep -qi "value=syncRev" $TMP_EXTRACT_DIR/templates/editor-preload-container.vm
        then
            echo "ERROR: Failed to update $TMP_EXTRACT_DIR/templates/editor-preload-container.vm"
            echo ""
            echo "UPDATE FAILED, EXITING!"
            exit 1;
        fi
        echo "ok"

        echo -n "   i. cleaning up temp files.."
        rm -f $TMP_EXTRACT_DIR/templates/editor-preload-container.vm $TMP_EXTRACT_DIR/templates/editor-preload-container.vm.bak $TMP_EXTRACT_DIR/templates/editor-preload-container.vm.original;
        rmdir $TMP_EXTRACT_DIR/templates 2> /dev/null;
        echo "ok"
    else
        echo "   b. templates/editor-preload-container.vm not present in JAR, skipping step"
    fi
else
    echo "   b. JAR not present in current install, skipping step"
fi

echo ""
echo "Update completed!"
